<?php
/**
 * TeachFrame - Cadriciel simple à finalité pédagogique pour l'enseignement du PHP.
 * Copyleft (c) 2024 Frank ENDRES (frank.endres@ac-polynesie.pf)
 * Ce programme est régi par la licence CeCILL 2.1 soumise au droit français et
 * respectant les principes de diffusion des logiciels libres.
 * http://www.cecill.info/licences/Licence_CeCILL_V2.1-fr.html
 */

namespace teachframe;
if (__FILE__ === $_SERVER['SCRIPT_FILENAME']) { header($_SERVER['SERVER_PROTOCOL'] . ' 403 Forbidden'); die(); }

/**
 * Cette classe gère les routes de l'application et  dispatche les requêtes
 * vers les contrôleurs appropriés en fonction de la méthode HTTP et de l'URL.
 * Les routes peuvent être chargées à partir d'un fichier YAML, ou ajoutées dans le code.
 */
class Router {
  private static string $baseDir;
  private array $routes;

  public function __construct() {
    self::autoSetBaseDir();
    $this->routes = [];
  }

  /**
   * Renvoie l'URL de la ressource corrigée du dossier racine.
   *
   * @param $resource l'URL de la ressource
   * @return le chemin du dossier racine
   */
  public static function getURL (string $resource): string {
    return self::$baseDir . $resource;
  }

  /**
   * Renvoie l'URI du referer **interne**.
   *
   * @return le referer ou '' s'il n'y en a pas, ou si externe
   */
  public static function getReferer (): string {
    $referer = '';
    if (isset($_SERVER['HTTP_REFERER']) && false !== strpos($_SERVER['HTTP_REFERER'], '/' . $_SERVER['HTTP_HOST'] . '/')) {
      $baseURL = (isset($_SERVER['HTTPS']) && "on" == $_SERVER['HTTPS'] ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'] . self::$baseDir;
      $referer = substr($_SERVER['HTTP_REFERER'], strlen($baseURL));
    }
    return $referer;
  }

  /**
   * Définit automatiquement le dossier de base à partir du dossier
   * du point d'entrée et de la racine des documents.
   */
  private static function autoSetBaseDir() {
    $dirName = dirname($_SERVER['SCRIPT_FILENAME']);
    $docRoot = $_SERVER['DOCUMENT_ROOT'];
    self::$baseDir = substr($dirName, strlen($docRoot));
    //~ if ('' === self::$baseDir && file_exists(dirname($_SERVER['SCRIPT_FILENAME']) . '/.htaccess')) {
      //~ $content = file_get_contents('.htaccess');
      //~ if (preg_match('/^RewriteBase\s+(.*)$/m', $content, $matches)) {
        //~ self::$baseDir = trim($matches[1]);
      //~ }
    //~ }
  }

  /**
   * Ajoute une nouvelle route.
   *
   * @param $route tableau associatif ayant les clés 'method' et 'path'
                   ainsi que la clé 'redirect' ou les clés 'controller', et 'action'.
   */
  public function addRoute (array $route) {
    $this->routes[] = $route;
  }

  /**
   * Charge les routes à partir d'un fichier YAML.
   *
   * @param $yamlFile le chemin vers le fichier YAML contenant les routes.
   * @end si le fichier n'existe pas, est mal formé ou mal structuré.
   *
   * Le fichier de routes doit être au format YAML et respecter la structure suivante :
   * routes:
   *   <route_name>:
   *     path: <route_path>
   *     method: <http_method>
   *     controller: <controller_class>
   *     action: <controller_method>
   */
  public function loadRoutes (string $yamlFile) {
    if (!file_exists($yamlFile)) {
      header($_SERVER['SERVER_PROTOCOL'] . ' 500 Internal Server Error');
      die('error: file \'' . $yamlFile . '\' is missing');
    }
    $content = yaml_parse_file($yamlFile);
    if (false === $content) {
      header($_SERVER['SERVER_PROTOCOL'] . ' 500 Internal Server Error');
      die('error: file \'' . $yamlFile . '\' cannot be parsed');
    }
    if (!isset($content['routes'])) {
      header($_SERVER['SERVER_PROTOCOL'] . ' 500 Internal Server Error');
      die('error: missing \'routes\' field in \'' . $yamlFile . '\'');
    }
    $this->routes = $content['routes'];
  }

  /**
   * Dispatche une requête en fonction de la méthode HTTP et de l'URL.
   *
   * @param $requestMethod méthode HTTP (GET, POST, ...).
   * @param $requestUri URL de la requête.
   * @end si une route est mal définie, ou si aucune route n'est trouvée pour la requête.
   */
  public function dispatch (string $requestMethod, string $requestUri) {
    $queryStringStart = strpos($requestUri, '?'); //présence facultative du '?'
    if ($queryStringStart !== false) {
      $requestUri = substr($requestUri, $queryStringStart + 1);
    }

    if (false !== strpos($requestUri, self::$baseDir)) {
      $requestUri = substr($requestUri, strlen(self::$baseDir));
    }
    $uriParts = explode('/', trim($requestUri, '/'));

    $found = false; foreach ($this->routes as $key => $route) {
      if (!isset($route['method']) || !isset($route['path'])) {
        header($_SERVER['SERVER_PROTOCOL'] . ' 500 Internal Server Error');
        die('error: route \'' . $key . '\' is missing \'method\' or \'path\'');
      }

      if ($route['method'] === $requestMethod) {
        $routeParts = explode('/', trim($route['path'], '/'));

        if (count($uriParts) === count($routeParts)) {
          $params = [];
          $match = true;

          foreach ($routeParts as $index => $part) { //vérification de chaque partie du chemin
            if (0 === strpos($part, '{') && strpos($part, '}') === strlen($part) - 1) {
              $params[] = $uriParts[$index];
            } elseif ($part !== $uriParts[$index]) {
              $match = false; break;
            }
          }

          if ($match) {
            if (isset($route['redirect'])) {
              header('Location: ' . self::$baseDir . $route['redirect']);
            } else if (isset($route['controller']) && isset($route['action'])) {
              $controller = new $route['controller']();
              $method = $route['action'];
              $controller->$method(...$params); //nombre variable de paramètres
            } else {
              header($_SERVER['SERVER_PROTOCOL'] . ' 500 Internal Server Error');
              die('error: route \'' . $key . '\' is missing \'redirect\' or \'controleur\' + \'action\'');
            }
            $found = true; break;
          }

        }
      }
    }
    if (!$found) {
      header($_SERVER['SERVER_PROTOCOL'] . ' 404 Not Found');
      die('error: route \'' . $requestMethod . ' ' . $requestUri . '\' not found');
    }
  }

  /**
   * Renvoie la documentation utilisateur des routes (pour API).
   *
   * @return tableau des routes.
   */
  public function getRoutes(): array {
    $routes = [];
    foreach ($this->routes as $route) {
      $fields = [];
      foreach ($route as $key => $value) {
        if ($key !== 'controller' && $key !== 'action' && $key !== 'redirect') {
          $fields[$key] = $value;
        }
      }
      $routes[] = $fields;
    }
    return $routes;
  }
}
