<?php
/**
 * TeachFrame - Cadriciel simple à finalité pédagogique pour l'enseignement du PHP.
 * Copyleft (c) 2024 Frank ENDRES (frank.endres@ac-polynesie.pf)
 * Ce programme est régi par la licence CeCILL 2.1 soumise au droit français et
 * respectant les principes de diffusion des logiciels libres.
 * http://www.cecill.info/licences/Licence_CeCILL_V2.1-fr.html
 */

namespace teachframe;
if (__FILE__ === $_SERVER['SCRIPT_FILENAME']) { header($_SERVER['SERVER_PROTOCOL'] . ' 403 Forbidden'); die(); }

class Cache {
  /**
   * Dossier de cache ; le serveur web doit avoir les permissions pour y écrire.
   */
  public static string $path = "cache";
  public static bool $disabled = false;

  /**
   * Réinitialise (supprime) un fichier ou un répertoire dans le cache
   *
   * @param $path le chemin du fichier ou répertoire à supprimer
   * @return true si la suppression a réussi, false sinon
   */
  public static function reset(string $filePath): bool {
    $cachePath = self::$path . DIRECTORY_SEPARATOR . $filePath;
    if (file_exists($cachePath)) {
        if (is_file($cachePath)) {
            return unlink($cachePath);
        } elseif (is_dir($cachePath)) {
          foreach (array_diff(scandir($cachePath), ['.', '..']) as $file) {
            if (!self::reset($filePath . DIRECTORY_SEPARATOR . $file)) return false; //don't use $cachePath !
          }
          return rmdir($cachePath);
        }
    }
    return false;
  }

  /**
   * Réinitialise (supprime) tous les fichiers du cache
   *
   * @return true si la suppression a réussi, false sinon
   */
  public static function resetAll(): bool {
    foreach (array_diff(scandir(self::$path), ['.', '..']) as $file) {
      if (!self::reset($file)) return false;
    }
    return true;
  }

  /**
   * Récupère le contenu d'un fichier depuis le cache.
   *
   * @param $path le chemin du fichier à récupérer
   * @return le contenu du fichier s'il existe, null sinon
   */
  public static function get(string $path): ?string {
    if (self::$disabled) return null;
    $path = self::$path . DIRECTORY_SEPARATOR . $path;
    if (file_exists($path) && is_file($path)) {
      return file_get_contents($path);
    }
    return null;
  }

  /**
   * Stocke du contenu dans un fichier du cache.
   *
   * @param $path Le chemin du fichier où stocker le contenu
   * @param $content Le contenu à stocker (chaîne de caractères ou autre type convertible)
   * @return true si le stockage a réussi, false sinon
   */
  public static function set(string $path, string $content): bool {
    if (self::$disabled) return true;
    $path = self::$path . DIRECTORY_SEPARATOR . $path;
    $dir = dirname($path);
    if (!is_dir($dir)) {
      if (!mkdir($dir)) return false;
      if (!chmod($dir, 0777)) return false;
    }
    if (!file_put_contents($path, $content)) return false;
    if (!chmod($path, 0777)) return false;
    return true;
  }
}
