<?php
/**
 * TeachFrame - Cadriciel simple à finalité pédagogique pour l'enseignement du PHP.
 * Copyleft (c) 2024 Frank ENDRES (frank.endres@ac-polynesie.pf)
 * Ce programme est régi par la licence CeCILL 2.1 soumise au droit français et
 * respectant les principes de diffusion des logiciels libres.
 * http://www.cecill.info/licences/Licence_CeCILL_V2.1-fr.html
**/
namespace teachframe;
if (__FILE__ === $_SERVER['SCRIPT_FILENAME']) { header($_SERVER['SERVER_PROTOCOL'] . ' 403 Forbidden'); die(); }

use \StdClass;
use \ReflectionClass;

class IO {
  /**
  * Fonction qui indique s'il y a des données en entrée associées à la requête
  * HTTP. Supporte les content-type x-www-form-urlencoded, multipart/form-data,
  * application/xml et application/json.
  * @return true s'il y a des données, ou dans le cas du JSON ou du XML.
  * @die si le format n'est pas supporté.
  **/
  public static function hasInput (): bool {
    if (!isset($_SERVER['CONTENT_TYPE'])) {
      $result = false;
    } else {
      if (  ('application/x-www-form-urlencoded' == $_SERVER['CONTENT_TYPE'] ||
             strpos($_SERVER['CONTENT_TYPE'], 'multipart/form-data') === 0)
          && ('GET' == $_SERVER['REQUEST_METHOD'] || 'POST' == $_SERVER['REQUEST_METHOD'])) {
        $assoc = 'GET' == $_SERVER['REQUEST_METHOD'] ? $_GET : $_POST;
        $result = 0 != count($assoc);
      } else if ('application/xml' == $_SERVER['CONTENT_TYPE'] || 'application/json' == $_SERVER['CONTENT_TYPE']) {
        $result = true;
      } else {
        header($_SERVER['SERVER_PROTOCOL'] . ' 400 Bad Request');
        die('error: unsupported content-type format \'' . $_SERVER['CONTENT_TYPE'] . '\'');
      }
    }
    return $result;
  }

  /**
  * Fonction de normalisation qui renvoie les données reçues par le serveur
  * sous la forme d'un objet, quelle que soit la méthode employée (GET, PUT
  * ou POST) et quel que soit le content-type (x-www-form-urlencoded, multipart/form-data,
  * application/xml et application/json).
  * @return un object contenant les données.
  * @die si le format n'est pas supporté ou si les données XML ou JSON sont mal formées.
  **/
  public static function getInput (): StdClass {
    if (!isset($_SERVER['CONTENT_TYPE'])) {
      header($_SERVER['SERVER_PROTOCOL'] . ' 400 Bad Request');
      die('error: unspecified content-type format');
    }
    if ('application/x-www-form-urlencoded' == $_SERVER['CONTENT_TYPE'] || 0 === strpos($_SERVER['CONTENT_TYPE'], 'multipart/form-data')) {
      $input = self::getFormInput();
    } else if ('application/xml' == $_SERVER['CONTENT_TYPE']) {
      $input = self::getXMLInput();
    } else if ('application/json' == $_SERVER['CONTENT_TYPE']) {
      $input = self::getJSONInput();
    } else {
        header($_SERVER['SERVER_PROTOCOL'] . ' 400 Bad Request');
        die('error: unsupported content-type format \'' . $_SERVER['CONTENT_TYPE'] . '\'');
    }
    return $input;
  }

  private static function getFormInput(): StdClass {
    if ('GET' == $_SERVER['REQUEST_METHOD'] || 'POST' == $_SERVER['REQUEST_METHOD']) {
      $assoc = 'GET' == $_SERVER['REQUEST_METHOD'] ? $_GET : $_POST; //$_REQUEST
    } else { //method should be 'PUT'
      if ('application/x-www-form-urlencoded' == $_SERVER['CONTENT_TYPE']) {
        parse_str(file_get_contents('php://input'), $assoc); //unlike POST, does not work with multipart/form-data
      } else {
        return self::parseMultipart();
      }
    }
    $input = new StdClass(); //associative array to object (normalization)
    foreach ($assoc as $key => $value) {
      self::setKey($input, $key, $value);
    }
    return $input;
  }

  private static function parseMultipart(): StdClass {
    if (false === strpos($_SERVER['CONTENT_TYPE'], 'boundary=')) {
      header($_SERVER['SERVER_PROTOCOL'] . ' 400 Bad Request');
      die('error: undefined boundary for multipart/form-data');
    }
    $bnd = '--' . explode('=', $_SERVER['CONTENT_TYPE'])[1];
    $bndlen = strlen($bnd);
    $txt = file_get_contents('php://input');
    $input = new StdClass();

    $start = $bndlen; $end = strpos($txt, $bnd, $start);
    while (false !== $end) {
      $qs = strpos($txt, '"', $start); $qe = strpos($txt, '"', $qs + 1);
      $key = substr($txt, $qs + 1, $qe - $qs - 1);
      $value = trim(substr($txt, $qe + 1, $end - $qe - 1));
      self::setKey($input, $key, $value);
      $start = $end + $bndlen; $end = strpos($txt, $bnd, $start);
    }
    return $input;
  }

  private static function setKey(StdClass &$input, string $key, mixed $value): void {
    if (isset($input->$key)) {
      if (is_array($input->$key)) {
        $input->$key[] = $value;
      } else {
        $input->$key = [ $input->$key, $value ];
      }
    } else {
      $input->$key = $value;
    }
  }

  private static function getXMLInput(): StdClass {
    $xml = simplexml_load_file('php://input');
    if (false === $xml) {
      header($_SERVER['SERVER_PROTOCOL'] . ' 400 Bad Request');
      die('error: malformed XML');
    }
    $input = new StdClass();
    $xml->rewind(); //mandatory
    for ($i=0; $i<count($xml); $i++) {
      $key = $xml->current()->getName();
      if ($xml->haschildren()) {
        foreach($xml->getChildren() as $value) { //NOTE: not recursive
          self::setKey($input, $key, (string) $value);
        }
      } else {
        $value = (string) $xml->current();
        self::setKey($input, $key, $value);
      }
      $xml->next();
    }
    return $input;
  }

  private static function getJSONInput(): StdClass {
    $input = json_decode(file_get_contents('php://input'));
    if (null === $input) {
      header($_SERVER['SERVER_PROTOCOL'] . ' 400 Bad Request');
      die('error: malformed JSON');
    }
    return $input;
  }

  /**
  * Valide les données en fonction des champs et types attendus.
  * @param fields la liste des champs attendus avec leur type ['champ:type', ...];
  *  - types: 's' (string), 'i' (int), 'f' (float), 'b' (bool), 'e' (email),
  *           'u' (url), 'd' (date), 't' (time);
  *  - préfixer le type avec '*' si une valeur est requise pour ce champ;
  *  - exemple: ['firstname:*s', 'email:e'].
  * @param input les données à valider (obtenues avec getInput).
  * @die si les données ne sont pas valides.
  **/
  public static function validateInput(array $fields, StdClass &$input): void {
    foreach ($fields as $field) {
      $field_type = explode(':', $field);
      $field = $field_type[0];
      $type = $field_type[1];
      if (isset($input->$field) && '' != $input->$field) {
        $input->$field = self::validate($input->$field, $type, $field);
      } else if ('*' == $type[0]) {
        header($_SERVER['SERVER_PROTOCOL'] . ' 400 Bad Request');
        die('error: missing required parameter \'' . $field . '\'');
      }
    }
  }

  private static function validate(string $str, string $type, string $field): mixed {
    $value = null;

    if ('*' == $type[0]) $type = substr($type, 1); //remove mendatory '*' indicator

    if ($type == 's') {
      $value = $str;
    } else if ($type == 'i') { //int validation
      $type = 'integer';
      $value = filter_var($str, FILTER_VALIDATE_INT, FILTER_NULL_ON_FAILURE);
    } else if ($type == 'f') { //float validation
      $type = 'float';
      $value = filter_var($str, FILTER_VALIDATE_FLOAT, FILTER_NULL_ON_FAILURE);
    } else if ($type == 'd') { //date validation
      $type = 'date (YYYY-MM-DD)';
      $value = date_create_from_format('Y-m-d', $str);
      if (false === $value) $value = null;
    } else if ($type == 't') { //time validation
      $type = 'time (HH:MM:SS)';
      $value = date_create_from_format('H:i:s', $str);
      if (false === $value) $value = null;
    } else if ($type == 'e') { //email validation
      $type = 'email';
      $value = filter_var($str, FILTER_VALIDATE_EMAIL, FILTER_NULL_ON_FAILURE);
    }  else if ($type == 'b') { //bool validation
      $type = 'boolean';
      $value = filter_var($str, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
    } else if ($type == 'u') { //url validation + sanitization
      $type = 'url';
      $value = filter_var($str, FILTER_VALIDATE_URL, FILTER_NULL_ON_FAILURE);
    }

    if (null === $value) {
      header($_SERVER['SERVER_PROTOCOL'] . ' 400 Bad Request');
      die('error: invalid ' . $str . ' value for parameter \'' . $field . '\': ' . $type . ' expected');
    }

    return $value;
  }


  public static string $default_output_format = 'application/json'; //or 'application/xml'
  /**
  * Renvoie des données sérialisées dans le format accepté: JSON ou XML.
  * @param output un object ou un tableau contenant des données.
  * @die si le format n'est pas supporté.
  **/
  public static function send(mixed $output): void {
    if (isset($_SERVER['HTTP_ACCEPT'])) {
      $accept = [];
      foreach (explode(',', $_SERVER['HTTP_ACCEPT']) as $dirtyAccept) {
        $accept[] = explode(';', $dirtyAccept)[0];
      }
    } else {
      $accept = [ self::$default_output_format ]; //normalization
    }
    if (in_array('application/json', $accept)) {
      header("Content-Type: application/json");
      ob_start();
      self::serializeToJSON($output);
      echo ob_get_clean();
    } else if (in_array('application/xml', $accept)) {
      header("Content-Type: application/xml"); //; charset=utf-8
      $xml = simplexml_load_string('<?xml version="1.0"?><data></data>');
      self::serializeToXML($xml, $output); //recursive
      echo $xml->asXML();
    } else {
      header($_SERVER['SERVER_PROTOCOL'] . ' 400 Bad Request');
      die('error: unsupported accept format \'' . $_SERVER['HTTP_ACCEPT'] . '\'');
    }
  }

  private static function serializeToXML(&$node, mixed $data) {
    if (is_iterable($data)) {
      foreach ($data as $key => $value) {
        if (is_int($key)) $key = 'item';
        if (is_array($value) || is_object($value)) {
          $subnode = $node->addChild($key);
          self::serializeToXml($subnode, $value);
        } else {
          $node->addChild($key, $value);
        }
      }
    } else if (is_object($data)) {
      $reflect = new ReflectionClass($data);
      $properties = $reflect->getProperties();
      foreach ($properties as $property) {
        if (!$property->isInitialized($data)) continue;
        $key = $property->getName();
        $value = $property->getValue($data);
        if (is_array($value) || is_object($value)) {
          $subnode = $node->addChild($key);
          self::serializeToXml($subnode, $value);
        } else {
          $node->addChild($key, $value);
        }
      }
    }
  }

  private static function serializeToJSON(mixed $data) { //json_encode does not work with non public properties
    if (is_iterable($data)) {
      $first = true;
      foreach ($data as $key => $value) {
        if ($first) {
          $first = false;
          echo is_int($key) ? '[' : '{'; //associative array to object
          $closeChar = is_int($key) ? ']' : '}';
        } else {
          echo ",";
        }
        if (!is_int($key)) echo '"' . $key . '":';
        if (is_array($value) || is_object($value)) {
          self::serializeToJSON($value);
        } else {
          echo is_bool($value) || is_int($value) || is_float($value) ? $value : '"' . $value . '"';
        }
      }
      echo /*empty*/ $first ? '[]' : $closeChar;
    } else if (is_object($data)) {
      $reflect = new ReflectionClass($data);
      $properties = $reflect->getProperties();
      echo '{';
      $first = true;
      foreach ($properties as $property) {
        if (!$property->isInitialized($data)) continue;
        $key = $property->getName();
        $value = $property->getValue($data);
        if ($first) $first = false; else echo ",";
        echo '"' . $key . '":';
        if (is_array($value) || is_object($value)) {
          self::serializeToJSON($value);
        } else {
          echo is_bool($value) || is_int($value) || is_float($value) ? $value : '"' . $value . '"';
        }
      }
      echo '}';
    }
  }
}
