<?php
/**
 * TeachFrame - Cadriciel simple à finalité pédagogique pour l'enseignement du PHP.
 * Copyleft (c) 2024 Frank ENDRES (frank.endres@ac-polynesie.pf)
 * Ce programme est régi par la licence CeCILL 2.1 soumise au droit français et
 * respectant les principes de diffusion des logiciels libres.
 * http://www.cecill.info/licences/Licence_CeCILL_V2.1-fr.html
 */

namespace teachframe;
if (__FILE__ === $_SERVER['SCRIPT_FILENAME']) { header($_SERVER['SERVER_PROTOCOL'] . ' 403 Forbidden'); die(); }

use \PDO;
use \ReflectionClass;
use \ReflectionProperty;

/**
 * Cette classe fournit des méthodes statiques pour interagir avec la base de données
 * en utilisant le mapping relationnel objet.
 * Contraintes:
 *  - la classe du modèle doit avoir le même nom que la table;
 *  - la clé primaire de la table doit être 'id'; pour les nouveaux enregistrements,
 *    id doit être initialisé à 0 si auto-incrémenté,
 *  - le nom des propriétés de la classe doit être identique à celui des champs;
 *  - associations de type OneToMany / ManyToOne (1-N / N-1):
 *      - l'objet associé doivent être de type 'NomClasseAssociée', et la
 *        clé étrangère doit être nommée 'idNomPropriété';
 *      - la collection associée doit avoir un attribut '#[ORM\ReversedBy\nomPropriété]'
 *        si la clé étrangère n'est pas nommée `idNomClasseAssociée',
 *  - les collections doivent être de type `NomClasseCollection`;
 *  - associations de type ManyToMany (N-N, avec table intermédiaire): ajouter
 *    l'attribut '#[ORM\Table\NomTableAssociation]';
 *  - en cas d'héritage, ne gère pas de table pour la classe parente
      (les champs hérités doivent être dans la table de la classe enfant).
 */
class ORM {
  static private ?PDO $pdo = null;

  /**
   * Initialise la connexion PDO à partir d'une connexion existante.
   *
   * @param $pdo la connexion existante
   * @end si le fichier n'existe pas, est mal formé ou mal structuré, ou si la connexion ne peut être établie.
   */
  public static function setPDO (PDO $pdo): void {
    self::$pdo = $pdo;
  }

  /**
   * Encapsule l'accès à la connexion PDO (pour s'assurer qu'elle est bien établie).
   *
   * @end si la connexion n'est pas établie.
   * @return la connexion PDO.
   */
  public static function getPDO(): PDO {
    if (null === self::$pdo) {
      header($_SERVER['SERVER_PROTOCOL'] . ' 500 Internal Server Error');
      die('error: PDO connection not established');
    }
    return self::$pdo;
  }

  /**
   * Initialise la connexion PDO à partir d'un fichier YAML.
   *
   * @param $configFile le chemin vers le fichier YAML contenant la configuration.
   * @end si le fichier n'existe pas, est mal formé ou mal structuré, ou si la connexion ne peut être établie.
   *
   * Le fichier de configuration doit être au format YAML et respecter la structure suivante :
   * db:
   *   dsn: <data source name>
   *   usr: <username>
   *   pwd: <password>
   */
  public static function setConfigPDO (string $configFile) {
    if (!file_exists($configFile)) {
      header($_SERVER['SERVER_PROTOCOL'] . ' 500 Internal Server Error');
      die('error: file \'' . $configFile . '\' is missing');
    }
    $config = yaml_parse_file($configFile);
    if (false === $config) {
      header($_SERVER['SERVER_PROTOCOL'] . ' 500 Internal Server Error');
      die('error: file \'' . $configFile . '\' cannot be parsed');
    }
    if (!isset($config['db']) || !isset($config['db']['dsn']) || !isset($config['db']['usr']) || !isset($config['db']['pwd'])) {
      header($_SERVER['SERVER_PROTOCOL'] . ' 500 Internal Server Error');
      die('error: missing \'db/dsn\', \'db/usr\' or \'db/pwd\' fields in \'' . $configFile . '\'');
    }
    try {
      self::$pdo = new PDO($config['db']['dsn'], $config['db']['usr'], $config['db']['pwd']);
    } catch(PDOException $e) {
      header($_SERVER['SERVER_PROTOCOL'] . ' 500 Internal Server Error');
      die($e->getmessage());
    }
  }


  /**
   * Récupère les enregistrements d'une table.
   *
   * @param $modelClass classe modèle (avec namespace).
   * @param $selectFields champs à récupérer (par défaut '*').
   * @param $filter clause WHERE optionnelle pour filtrer les résultats.
   * @param $offset pour renvoyer un sous-ensemble des enregistrements.
   * @param $limit pour renvoyer un sous-ensemble des enregistrements.
   * @return tableau d'objets récupérés, sans chargement des objets associés.
   * ATTENTION les chaînes $selectFields et $filter ne sont pas échappées.
   */
  public static function getAll (string $modelClass, string $selectFields = '*',
                 ?string $filter = null, int $offset = 0, int $limit = 0): array {
  $limit = 0 !== $offset && 0 !== $limit ? ' LIMIT ' . $limit . ' OFFSET ' . $offset : '';
    $where = null !== $filter ? ' WHERE ' . $filter : '';
    $stmt = self::getPDO()->query('SELECT ' . $selectFields . ' FROM ' . self::getBaseName($modelClass) . $where . $limit);
    $collection = [];
    while ($item = $stmt->fetch(PDO::FETCH_ASSOC)) {
      $collection[] = self::buildObj($modelClass, $item);
    }
    return $collection;
  }

  /**
   * Récupère un enregistrement d'une table à partir de son identifiant.
   *
   * @param $modelClass classe modèle (avec namespace).
   * @param $id identifiant de l'enregistrement à récupérer.
   * @param $followLinks pour charger (par défaut) ou pas les données des objets liés.
   * @return objet récupéré avec chargement des objets associés ou null si non trouvé.
   */
  public static function getOne (string $modelClass, mixed $id, bool $followLinks = true): mixed {
    $filter = 'id = ' . (!is_int($id) ? self::getPDO()->quote($id) : $id);
    $stmt = self::getPDO()->query('SELECT * FROM ' . self::getBaseName($modelClass) . ' WHERE ' . $filter);
    $obj = null;
    if ($item = $stmt->fetch(PDO::FETCH_ASSOC)) {
      $obj = self::buildObj($modelClass, $item, $followLinks);
    }
    return $obj;
  }


  /**
   * Ajoute un enregistrement.
   *
   * @param $obj objet à insérer.
   * @return true si un objet a été créé, false sinon.
   */
  public static function insert (mixed $obj): bool {
    return self::persist($obj, 'insert');
  }

  /**
   * Modifie un enregistrement.
   *
   * @param $obj objet à mettre à jour.
   * @return true si un objet a été modifié, false sinon.
   */
  public static function update (mixed $obj): bool {
    return self::persist($obj, 'update');
  }

  /**
   * Ajoute ou modifie un enregistrement.
   *
   * @param $obj objet à insérer ou mettre à jour.
   * @param $operation l'opération SQL ('insert' ou 'update') - autodétectée par défaut.
   * @return true si un objet a été bien été créé ou modifié, false sinon.
   */
  public static function persist (mixed $obj, ?string $operation = null): bool {
    $table = self::getBaseName(get_class($obj));
    $columns = []; $values = []; $updates = []; $collections = [];
    $reflect = new ReflectionClass($obj);
    $modelClass = $reflect->getName();
    foreach ($reflect->getProperties() as $property) {
      $propertyType = (string) $property->getType();
      $propertyName = $property->getName();
      $propertyValue = $property->getValue($obj);
      if ('id' !== $propertyName) {
        switch ($propertyType) {
          case 'bool':
          case 'int':
          case 'float':
            $columns[] = $propertyName;
            $values[] = $propertyValue;
            $updates[] = $columns[count($columns) - 1] . ' = ' . $values[count($columns) - 1];
            break;
          case 'string':
          case 'DateTime':
            $columns[] = $propertyName;
            $values[] = self::getPDO()->quote((string) $propertyValue);
            $updates[] = $columns[count($columns) - 1] . ' = ' . $values[count($columns) - 1];
            break;
          default: //is_object($propertyValue))
            if (str_ends_with($propertyType, 'Collection')) {
              $collections[] = $property;
            } else {
              $columns[] = 'id' . ucfirst($propertyName);
              $values[] = $propertyValue->getId();
              $updates[] = $columns[count($columns) - 1] . ' = ' . $values[count($columns) - 1];
            }
        }
      }
    }

    if (null === $operation) {
      $operation = null == self::getOne($modelClass, $obj->getId(), false) ? 'insert' : 'update';
    }
    if ('insert' === $operation) {
      $sql = 'INSERT INTO ' . $table . '(' . implode(', ', $columns) . ') VALUES (' . implode(', ', $values) . ')';
    } else {
      $filter = 'id = ' . (!is_int($obj->getId()) ? self::getPDO()->quote($obj->getId()) : $obj->getId());
      $sql = 'UPDATE ' . $table . ' SET ' . implode(', ', $updates) . ' WHERE ' . $filter;
    }
    $result = self::getPDO()->exec($sql);
    if (1 != $result) return false;

    if (0 === $obj->getId()) { //INSERT + AUTO_INCREMENT
      $obj->setId(self::getPDO()->lastInsertId());
    }

    foreach($collections as $property) { //persist ManyToMany (N-N) subCollection
      $attributes = array_filter($property->getAttributes(), fn($attr) => str_contains($attr->getName(), 'ORM\\Table\\'));
      if (1 === count($attributes)) {
        $internClass = self::getBaseName($attributes[0]->getName());
        $externClassField = 'id' . self::getBaseName($modelClass);
        if ('update' === $operation) {
          $filter = $externClassField . ' = ' . (!is_int($obj->getId()) ? self::getPDO()->quote($obj->getId()) : $obj->getId());
          self::getPDO()->exec('DELETE FROM ' . self::getBaseName($internClass) . ' WHERE ' . $filter);
        }
        $externRelatedClassField = 'id' . self::getBaseName(substr((string) $property->getType(), 0, -10));
        foreach($property->getValue($obj) as $relatedObj) {
          self::getPDO()->exec('INSERT INTO ' . self::getBaseName($internClass) . '(' . $externClassField . ', ' . $externRelatedClassField . ') ' .
                               'VALUES (' . $obj->getId() . ', ' . $relatedObj->getId() .')' . "\n");
        }
      }
    }

    return true;
  }


  /**
   * Supprime un enregistrement.
   *
   * @param $obj objet à supprimer.
   * @return true si un objet a été supprimé, false sinon.
   *
   */
  public static function delete (mixed $obj): bool {
    $reflect = new ReflectionClass($obj);
    foreach ($reflect->getProperties() as $property) {
      $propertyType = (string) $property->getType();
      $propertyName = $property->getName();
      if (str_ends_with($propertyType, 'Collection')) {
        $attributes = array_filter($property->getAttributes(), fn($attr) => str_contains($attr->getName(), 'ORM\\Table\\'));
        if (1 === count($attributes)) { //if there is an intermediate table - cf ManyToMany (N-N)
          $internClass = self::getBaseName($attributes[0]->getName());
          $modelClass = $reflect->getName();
          $externClassField = 'id' . self::getBaseName($modelClass);
          self::getPDO()->exec('DELETE FROM ' . self::getBaseName($internClass) . ' ' .
                               'WHERE ' . $externClassField . ' = ' . $obj->getId());
        }
      }
    }
    $table = self::getBaseName (get_class($obj));
    $filter = 'id = ' . (!is_int($obj->getId()) ? self::getPDO()->quote($obj->getId()) : $obj->getId());
    $result = self::getPDO()->exec('DELETE FROM ' . $table . ' WHERE ' . $filter);
    return 1 == $result;
  }

  /**
   * Supprime plusieurs enregistrement d'une table.
   *
   * @param $modelClass classe modèle (avec namespace).
   * @param $filter clause WHERE pour sélectionner les enregistrements à supprimer.
   * ATTENTION la chaîne $filter n'est pas échappée.
   */
  public static function deleteMany (string $modelClass, string $filter): void {
    self::getPDO()->exec('DELETE FROM ' . self::getBaseName($modelClass) . ' WHERE ' . $filter);
  }


  private static function buildObj (string $modelClass, array $item, bool $followLinks = false) {
    $reflect = new ReflectionClass($modelClass);
    $properties = $reflect->getProperties();
    $obj = $reflect->newInstanceWithoutConstructor(); ////generates the obj(ect) from the item
    foreach ($properties as $property) { //by copying each property
      $propertyName = $property->getName();
      $propertyType = (string) $property->getType();
      switch ($propertyType) {
        case 'bool':
        case 'int':
        case 'float':
        case 'string':
          $property->setValue($obj, $item[$propertyName]);
          break;
        case 'DateTime':
          $property->setValue($obj, date_create_from_format('Y-m-d', $item[$propertyName]));
          break;
        default: //is_object($propertyValue)
          if ($followLinks) {
            if (str_ends_with($propertyType, 'Collection')) {
              self::setSubCollection($modelClass, $item, $obj, $property);
            } else {
              self::setSubObj($item, $obj, $property);
            }
          }
      }
    }
    return $obj;
  }

  private static function setSubObj (array $item, mixed $obj, ReflectionProperty $property): void {
    $externClassField = 'id' . ucfirst($property->getName());
    if (null !== $item[$externClassField]) {
      $externObj = self::getOne((string) $property->getType(), $item[$externClassField], false); //cut recursivity
      assert(null !== $externObj, 'DBMS integrity constraint error');
      $property->setValue($obj, $externObj);
    } else {
      $property->setValue($obj, null);
    }
  }

  private static function setSubCollection (string $modelClass, array $item, mixed $obj, ReflectionProperty $property): void {
    $typeName = (string) $property->getType();
    $externClass = substr($typeName, 0, -10);
    $externClassField = self::getExternClassField($modelClass, $property);
    $attributes = array_filter($property->getAttributes(), fn($attr) => str_contains($attr->getName(), 'ORM\\Table\\'));
    if (1 === count($attributes)) { //if there is an intermediate table - cf ManyToMany (N-N)
      $internClass = self::getBaseName($attributes[0]->getName());
      $stmt = self::getPDO()->query('SELECT id' . self::getBaseName($externClass) . ' FROM ' . $internClass . ' WHERE ' . $externClassField . ' = ' . $obj->getId());
      $filter = 'id IN ' . '(' . implode(', ',  $stmt->fetchAll(PDO::FETCH_COLUMN, 0)) . ')';
    } else { //OneToMany (1-N)
      $filter = $externClassField . ' = ' . (!is_int($obj->getId()) ? self::getPDO()->quote($obj->getId()) : $obj->getId());
    }
    $items = self::getAll($externClass, '*', $filter);
    $property->setValue($obj, new $typeName($items));
  }

  private static function getExternClassField (string $modelClass, ReflectionProperty $property): string {
    $attributes = array_filter($property->getAttributes(), fn($attr) => str_contains($attr->getName(), 'ORM\\ReversedBy\\'));
    if (1 === count($attributes)) { //use metadata if specified
      $externClassField = 'id' . ucFirst(self::getBaseName($attributes[0]->getName()));
    } else { //otherwise use convention
      $externClassField = 'id' . self::getBaseName($modelClass);
    }
    return $externClassField;
  }

  private static function getBaseName (string $modelClass): string { // 'model\SomeClass' -> 'SomeClass'
    $classPath = explode('\\', $modelClass);
    return $classPath[count($classPath) - 1];
  }

  //~ private static function getDirName (string $modelClass): string { // 'org\app\model\SomeClass' -> 'org\app\model'
    //~ return join('\\', array_slice(explode('\\', $modelClass), 0, -1));
  //~ }
}
