<?php
namespace controller;
use teachframe\ORM;
use teachframe\Cache;
use teachframe\IO;
use teachframe\Router;
use model\Island;
use model\Archipelago;
use view\IslandView;

class IslandController {
  function getAll() {
    $html = Cache::get('/islands');
    if (null == $html) {
      $islands = ORM::getAll(Island::class);
      $html = new IslandView()->renderAll($islands);
      Cache::set('/islands', $html);
    }
    echo $html;
  }

  function getOne (string $id) {
    $html = Cache::get('/island/' . $id);
    if (null == $html) {
      $island = $this->initIsland($id);
      $html = new IslandView()->renderOne($island);
      Cache::set('/island/' . $id, $html);
    }
    echo $html;
  }

  function initIsland(int $id): Island {
    if (0 == $id) { //pour une nouvelle île
      $island = new Island(0, '', -1, -1, new Archipelago(0, ''));
    } else { //pour une île existante
      $island = ORM::getOne(Island::class, $id);
      if (null == $island) { //si l'île n'existe pas
        http_response_code(404);
        die('error: \'/island/' . $id . '\' not found');
      }
    }
    return $island;
  }

  function edit (string $id) {
    $island = $this->initIsland(intval($id));
    if (0 == $id) {
      $referer = explode('/', Router::getReferer());
      $idArchipelago = 3 <= count($referer) && 'archipelago' == $referer[1] ? intval($referer[2]) : 0;
      $island->setArchipelago(new Archipelago($idArchipelago, ''));
    }
    $html = new IslandView()->renderEdit($island, ORM::getAll(Archipelago::class)); //rendu du formulaire
    echo $html;
  }

  function save(string $id) {
    $island = $archipelago = $this->initIsland(intval($id));
    if (IO::hasInput()) {
      $input = IO::getInput();
      IO::validateInput(['name:*s', 'population:*i', 'area:*i', 'idArchipelago:*i'], $input);
      $archipelago = ORM::getOne(Archipelago::class, $input->idArchipelago, false);
      if (null == $archipelago) {
        http_response_code(400);
        die('error: integrity constraint failed (idArchipelago = ' . $input->idArchipelago . ')');
      }
      $island->setName($input->name);
      $island->setPopulation($input->population);
      $island->setArea($input->area);
      $island->setArchipelago($archipelago);
      ORM::persist($island, 0 == $id ? 'insert' : 'update');
    } else if (0 != $id) { //sans données, c'est une suppression
      ORM::delete($island);
    } else {
      http_response_code(400);
      die('error: cannot delete \'/island/0\'');
    }
    //Cache::resetAll();
    Cache::reset('/islands');
    Cache::reset('/island/' . $island->getId());
    Cache::reset('/archipelago/' . $island->getArchipelago()->getId());
    header('Location: ' . Router::getURL('/archipelago/' . $island->getArchipelago()->getId()));
  }
}
